/*
 * Copyright 2003 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/tree.h>
#include <sys/queue.h>

#ifndef WIN32
#include <sys/param.h>
#include <unistd.h>
#endif
#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <dnet.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "subsystem.h"
#include "fdpass.h"

ssize_t atomicio(ssize_t (*)(), int, void *, size_t);

void subsystem_read(int, short, void *);
void subsystem_write(int, short, void *);

struct callback subsystem_cb = {
	subsystem_read, subsystem_write, NULL, NULL
};

/* Determine if the socket information is valid */

int
subsystem_socket(struct subsystem_command *cmd, int local,
    char *ip, size_t iplen, u_short *port, int *proto)
{
	struct sockaddr_in *si;
	struct addr src;
	socklen_t len;

	si = (struct sockaddr_in *)(local ? &cmd->sockaddr : &cmd->rsockaddr);
	len = local ? cmd->len : cmd->rlen;

	/* Only IPv4 TCP or UDP is allowed.  No raw sockets or such */
	if (si->sin_family != AF_INET || cmd->domain != AF_INET ||
	    !(cmd->type == SOCK_DGRAM || cmd->type == SOCK_STREAM) ||
	    len != sizeof(struct sockaddr_in))
		return (-1);

	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &si->sin_addr.s_addr,
	    IP_ADDR_LEN);
	addr_ntop(&src, ip, iplen);

	*port = ntohs(si->sin_port);
	*proto = cmd->type == SOCK_DGRAM ? IP_PROTO_UDP : IP_PROTO_TCP;

	return (0);
}

void
subsystem_cleanup(struct subsystem *sub)
{
	struct port *port;

	syslog(LOG_INFO, "Subsystem \"%s\" died", sub->cmdstring);

	for (port = TAILQ_FIRST(&sub->ports); port;
	    port = TAILQ_FIRST(&sub->ports)) {

		/* Free all ports for this subsystem */
		port_free(sub->tmpl, port);
	}

	/* XXX - do proper cleanup here */
}

void
subsystem_readyport(struct port *port, struct subsystem *sub)
{
	port->sub = sub;
	port->sub_fd = -1;
	TAILQ_INSERT_TAIL(&sub->ports, port, next);
}

void
subsystem_read(int fd, short what, void *arg)
{
	struct subsystem *sub = arg;
	struct template *tmpl = sub->tmpl;
	struct subsystem_command cmd;
	struct sockaddr_in *si = (struct sockaddr_in *)&cmd.sockaddr;
	char asrc[24];
	u_short port;
	int proto;
	char res = -1;

	if (atomicio(read, fd, &cmd, sizeof(cmd)) != sizeof(cmd)) {
		subsystem_cleanup(sub);
		return;
	}

	switch (cmd.command) {
	case BIND: {
		struct port *sub_port;
		struct action action;
	
		/* Check address family */
		if (subsystem_socket(&cmd, 1, asrc, sizeof(asrc),
			&port, &proto) == -1)
			goto out;

		/* See if it tries to bind an address that we know */
		if (si->sin_addr.s_addr != IP_ADDR_ANY &&
		    strcmp(tmpl->name, asrc)) {
			syslog(LOG_WARNING,
			    "Subsystem %s on %s attempts illegal bind %s:%d",
			    sub->cmdstring, tmpl->name, asrc, port);
			goto out;
		}

		/* Setup port type */
		memset(&action, 0, sizeof(action));
		action.status = PORT_RESERVED;

		sub_port = port_insert(tmpl, proto, port, &action);
		if (sub_port == NULL)
			goto out;
		
		/* Set up necessary port information */
		subsystem_readyport(sub_port, sub);

		syslog(LOG_DEBUG, "Subsytem \"%s\" binds %s:%d",
		    sub->cmdstring, tmpl->name, port);

		res = 0;

		break;
	}

	case LISTEN: {
		struct port *sub_port;
		int nfd;

		/* Check address family */
		if (subsystem_socket(&cmd, 1, asrc, sizeof(asrc),
			&port, &proto) == -1) {
			syslog(LOG_WARNING, "%s: listen bad socket", __func__);
			goto out;
		}

		/* find bound port */
		sub_port = port_find(tmpl, proto, port);
		if (sub_port == NULL || sub_port->sub != sub) {
			syslog(LOG_WARNING, "%s: proto %d port %d not bound",
			    __func__, proto, port);
			goto out;
		}

		res = 0;
		atomicio(write, fd, &res, 1);
		res = -1;

		/* Repeat until we get a result */
		while ((nfd = receive_fd(fd, NULL, NULL)) == -1) {
			if (errno != EAGAIN)
				break;
		}

		if (nfd == -1) {
			syslog(LOG_WARNING, "%s: no file descriptor",__func__);
			goto out;
		}

		syslog(LOG_DEBUG, "Listen: %s:%d -> fd %d", asrc, port, nfd);

		/* We use this fd to notify the other side */
		sub_port->sub_fd = nfd;
		sub_port->sub_islisten = 1;

		/* Enable this port */
		sub_port->action.status = PORT_SUBSYSTEM;

		res = 0;
		break;
	}

	case CLOSE: {
		struct port *sub_port;

		/* Check address family */
		if (subsystem_socket(&cmd, 1, asrc, sizeof(asrc),
			&port, &proto) == -1)
			goto out;

		syslog(LOG_DEBUG, "Close: %s:%d", asrc, port);

		/* XXX - only bound port */
		sub_port = port_find(tmpl, proto, port);
		if (sub_port == NULL || sub_port->sub != sub)
			goto out;

		port_free(tmpl, sub_port);

		break;
		}

	case CONNECT: {
		struct port *sub_port;
		struct action action;
		struct addr src, dst;
		struct ip_hdr ip;

		/* Check remote address family */
		if (subsystem_socket(&cmd, 0, asrc, sizeof(asrc),
			&port, &proto) == -1)
			goto out;
		
		syslog(LOG_DEBUG, "Connect: %s %s:%d",
		    proto == IP_PROTO_UDP ? "udp" : "tcp", asrc, port);

		if (addr_aton(tmpl->name, &src) == -1)
			goto out;
		if (addr_aton(asrc, &dst) == -1)
			goto out;

		memset(&action, 0, sizeof(action));
		action.status = PORT_RESERVED;

		sub_port = port_random(tmpl, proto, &action, 1024, 49151);
		if (sub_port == NULL)
			goto out;
		syslog(LOG_DEBUG, "Connect: allocated port %d",
		    sub_port->number);

		subsystem_readyport(sub_port, sub);

		/* The remote side is the source */
		ip.ip_src = dst.addr_ip;
		ip.ip_dst = src.addr_ip;

		/* Try to setup a TCP connection */
		if (proto == IP_PROTO_TCP) {
			struct tcp_con *con;
			struct tcp_hdr tcp;
			int nfd;

			tcp.th_sport = htons(port);
			tcp.th_dport = htons(sub_port->number);

			if ((con = tcp_new(&ip, &tcp, 1)) == NULL)
				goto out;
			con->tmpl = tmpl;

			/* Cross notify */
			con->port = sub_port;
			sub_port->sub_conport = &con->port;

			/* Confirm success of this phase */
			res = 0;
			atomicio(write, fd, &res, 1);
			
			/* Now get the control fd */
			while ((nfd = receive_fd(fd, NULL, NULL)) == -1) {
				if (errno != EAGAIN) {
					tcp_free(con);
					goto out;
				}
			}
			sub_port->sub_fd = nfd;

			/* Confirm success again */
			res = 0;
			atomicio(write, nfd, &res, 1);
			
			/* Send out the SYN packet */
			con->state = TCP_STATE_SYN_SENT;
			tcp_send(con, TH_SYN, NULL, 0);
			con->snd_una++;

			con->retrans_time = 1;
			generic_timeout(&con->retrans_timeout, con->retrans_time);
			goto reschedule;
		} else if (proto == IP_PROTO_UDP) {
			struct udp_con *con;
			struct udp_hdr udp;

			/* The remote side is the source */
			udp.uh_sport = htons(port);
			udp.uh_dport = htons(sub_port->number);

			if ((con = udp_new(&ip, &udp, 1)) == NULL)
				goto out;
			con->tmpl = tmpl;
			
			/* Cross notify */
			con->port = sub_port;
			sub_port->sub_conport = &con->port;

			/* Confirm success of this phase */
			res = 0;
			atomicio(write, fd, &res, 1);

			/* Connect our system to the subsystem */
			cmd_subsystem_localconnect(&con->conhdr, &con->cmd,
			    sub_port, con);
			goto reschedule;
		}
	}
	default:
		break;
	}

 out:
	atomicio(write, fd, &res, 1);
 reschedule:
	/* Reschedule read */
	event_add(&sub->cmd.pread, NULL);
}

void
subsystem_write(int fd, short what, void *arg)
{
	/* Nothing */
}
