/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef _HONEYD_H_
#define _HONEYD_H_

#ifndef timeout_add
#define timeout_add(ev, tv)             event_add(ev, tv)
#define timeout_set(ev, cb, arg)        event_set(ev, -1, 0, cb, arg)
#define timeout_del(ev)                 event_del(ev)
#define timeout_pending(ev, tv)         event_pending(ev, EV_TIMEOUT, tv)
#define timeout_initialized(ev)         ((ev)->ev_flags & EVLIST_INIT)
#endif

#ifdef WIN32
#define PIDFILE			"honeyd.pid"
#define PATH_HONEYDDATA "."
#define PATH_HONEYDLIB "."
#else
#define PIDFILE			"/var/run/honeyd.pid"
#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#endif

#define TCP_DEFAULT_SIZE	128
#define TCP_MAX_SIZE		4096
#define TCP_MAX_SEND		512

#define HONEYD_CLOSE_WAIT	10
#define HONEYD_SYN_WAIT		20
#define HONEYD_MAX_ACTIVE	(60*60*1)
#define HONEYD_DFL_TTL		64
#define HONEYD_UDP_WAIT		60
#define HONEYD_MAX_SOFTERRS	3	   /* Softerrors to state free*/

#define HONEYD_POLL_INTERVAL	{0, 10000}

#define HONEYD_ADDR_MASK        0xFFFFFF00 /* for ICMP address mask replies */

struct delay {
	struct event timeout;

	int ttl;
	struct addr host;
	struct template *tmpl;
};

enum status {PORT_OPEN = 0, PORT_PROXY, PORT_BLOCK, PORT_RESET,
	     PORT_SUBSYSTEM, PORT_RESERVED
};

#define PORT_ISOPEN(x) ((x)->status == PORT_OPEN || \
			(x)->status == PORT_PROXY || \
			(x)->status == PORT_SUBSYSTEM)			

struct subsystem;
struct action {
	char *action;
	struct addrinfo *aitop;
	enum status status;
};

struct port {
	SPLAY_ENTRY(port) node;
	TAILQ_ENTRY(port) next;

	int proto;
	u_short number;

	struct action action;

	/* Subsystem related information */
	struct subsystem *sub;
	int sub_fd;
	int sub_token;		/* identifier for the other side */
	int sub_islisten;
	struct port **sub_conport;
};

SPLAY_HEAD(porttree, port);

struct personality;
struct subsystem;
struct template {
	SPLAY_ENTRY(template) node;

	char *name;

	struct porttree ports;

	struct action icmp;
	struct action tcp;
	struct action udp;

	struct personality *person;

	int id;
	uint32_t seq;
	int seqcalls;
	uint32_t timestamp;
	struct timeval tv;

	uint16_t drop_inrate;

	uid_t uid;
	gid_t gid;

	TAILQ_HEAD(subsysqueue, subsystem) subsystems;
};

/* Contains information common to both UDP and TCP connections */

struct tuple {
	SPLAY_ENTRY(tuple) node;

	ip_addr_t ip_src;
	ip_addr_t ip_dst;
	uint16_t sport;
	uint16_t dport;

	int type;	/* Eiter SOCK_STREAM or SOCK_DGRAM */

	/* Statistics */
	uint32_t received;
	uint32_t sent;

	int local;	/* locally initiated */
};

struct command {
	pid_t pid;
#ifdef WIN32
	int pwritefd;
	int IsProxyCmd;
#endif
	int pfd;
	int perrfd;

	struct event pread;
	struct event pwrite;
	struct event peread;

	uint8_t fdconnected:1,
	        fdwantclose:1,
	        unused:6;
};

/* Subsystem state */

struct subsystem {
	TAILQ_ENTRY(subsystem) next;

	struct template *tmpl;		/* back pointer to parent */
	char *cmdstring;

	struct command cmd;

	TAILQ_HEAD(portqueue, port) ports;	/* list of configured ports */
};

/* State about TCP connections */

struct tcp_con {
	/* Has to be the first member of the structure */
	struct tuple conhdr;
#define con_ipsrc conhdr.ip_src
#define con_ipdst conhdr.ip_dst
#define con_sport conhdr.sport
#define con_dport conhdr.dport

	uint8_t dupacks;
	uint32_t snd_una;

	uint32_t rcv_next;
	uint32_t last_acked;

	struct template *tmpl;
	uint8_t rcv_flags;

	struct command cmd;
#define cmd_pfd	cmd.pfd
#define cmd_perrfd cmd.perrfd

	u_char *payload;
	u_int psize;
	u_int plen;		/* date in buffer */
	u_int poff;		/* current send offset */

	u_char *readbuf;
	u_int rsize;
	u_int rlen;

	uint8_t state;
	uint8_t sentfin:1,
		finacked:1,
		sawwscale:1,
		sawtimestamp:1,
		unused:4;

	u_short	mss;
	u_short window;
	uint32_t echotimestamp;

	u_short retrans_time;

	struct event timeout;
	struct event retrans_timeout;

	struct port *port;
};

#define MAX_UDP_BUFFERS	10

struct buffer {
	TAILQ_ENTRY(buffer) next;

	u_char *buf;
	size_t len;
};

struct udp_con {
	/* Has to be the first member of the structure */
	struct tuple conhdr;

	struct template *tmpl;

	struct command cmd;

	TAILQ_HEAD(bufferq, buffer) incoming;
	int nincoming;

	struct event timeout;

	int softerrors;		/* ICMP unreachables for this state */

	struct port *port;
};

struct callback {
	void (*cb_read)(int, short, void *);
	void (*cb_write)(int, short, void *);
	void (*cb_eread)(int, short, void *);
	void (*cb_connect)(int, short, void *);
};

/* YM
 * Timestamp message data
 */
struct icmp_msg_timestamp { /* dnet.h define this but the size is wrong */
				/* So define ours */
	struct icmp_hdr hdr;			/* ICMP header */
	uint16_t        icmp_id;                /* identifier */
        uint16_t        icmp_seq;               /* sequence number */
        uint32_t        icmp_ts_orig;           /* originate timestamp */
        uint32_t        icmp_ts_rx;             /* receive timestamp */
        uint32_t        icmp_ts_tx;             /* transmit timestamp */
};

/* YM
 * Address mask message data, RFC 950
 */
struct icmp_mesg_mask { /* Our definition */
	struct icmp_hdr hdr;			/* ICMP header */
        uint16_t        icmp_id;                /* identifier */
        uint16_t        icmp_seq;               /* sequence number */
        uint32_t        icmp_mask;              /* address mask */
};

/* YM
 * Information Reply message data, RFC 792
 */
struct icmp_msg_inforeply { /* Our definition */
	struct icmp_hdr hdr;			/* ICMP header */
	struct icmp_msg_idseq idseq;		/* ID_SEQ */
};

#define TCP_BYTESINFLIGHT(x)	(x)->poff
#define TCP_MAX_INFLIGHT	4096

void honeyd_dispatch(struct template *, struct ip_hdr *);
char *honeyd_contoa(struct tuple *);

int honeyd_logstart(char *);
void honeyd_logend(void);
void honeyd_log_probe(int, struct tuple *, int, int);
void honeyd_log_flownew(int, struct tuple *);
void honeyd_log_flowend(int, struct tuple *);

/* Command prototypes for services */

void cmd_ready_fd(struct command *, struct callback *, void *);
void cmd_trigger_read(struct command *, int);
void cmd_trigger_write(struct command *, int);
void cmd_free(struct command *);
int cmd_fork(struct tuple *, struct command *, struct template *,
    char *, char **, void *);
int cmd_subsystem(struct template *, struct subsystem *, char *, char **);

struct addrinfo;
struct addrinfo *cmd_proxy_getinfo(char *, int, short);
int cmd_proxy_connect(struct tuple *, struct command *, struct addrinfo *,
    void *);
int cmd_subsystem_connect(struct tuple *hdr, struct command *cmd,
    struct port *, void *arg);
int cmd_subsystem_localconnect(struct tuple *hdr, struct command *cmd,
    struct port *, void *arg);

/* Network connection elements */
struct tcp_con *tcp_new(struct ip_hdr *, struct tcp_hdr *, int);
struct udp_con *udp_new(struct ip_hdr *, struct udp_hdr *, int);
int tcp_setupconnect(struct tcp_con *);

void generic_timeout(struct event *, int);

/* Network protocol related prototypes */

void tcp_free(struct tcp_con *);
int tcp_send(struct tcp_con *, uint8_t, u_char *, u_int);
void tcp_senddata(struct tcp_con *, uint8_t);
void tcp_sendfin(struct tcp_con *);

void udp_free(struct udp_con *);
int udp_send(struct udp_con *con, u_char *payload, u_int len);

void config_init(void);
struct template *template_find(char *);
struct port *port_insert(struct template *, int, int, struct action *);
struct port *port_random(struct template *, int, struct action *, int, int);
struct port *port_find(struct template *, int, int);
void port_free(struct template *, struct port *);

int parse_configuration(FILE *, char *);

char *strrpl(char *, size_t, char *, char *);

void icmp_echo_reply(struct ip_hdr *rip, uint8_t code, uint16_t ipid,
		uint8_t tos, uint16_t offset, uint8_t ttl,
		u_char *payload, u_int len);
void change_quote_header(struct ip_hdr *rip, uint16_t udp_cksum, 
		uint16_t ip_cksum, uint16_t ipid, uint16_t iplen, 
		uint16_t offset);
void icmp_unreachable_reply(struct ip_hdr *rip, uint8_t ttl, uint8_t tos,
	uint16_t df, uint16_t riplen);
void icmp_mask_reply(struct ip_hdr *rip, 
	struct icmp_msg_idseq *idseq, uint8_t ttl, uint32_t addrmask);
void icmp_info_reply(struct ip_hdr *rip, 
		struct icmp_msg_idseq *idseq, uint8_t ttl);
void icmp_timestamp_reply(struct ip_hdr *rip,
    struct icmp_msg_timestamp* icmp_rip, uint8_t ttl);

#endif
