/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_IOCCOM_H
#include <sys/ioccom.h>
#endif
#ifdef WIN32
#include "getopt.h"
#include "misc.h"
#include "packet32.h"
#include "NtDDNdis.h"
#else
#include <sys/param.h>
#include <sys/resource.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <sys/wait.h>
#endif
#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/queue.h>

#ifdef WIN32
#define BPF_MAJOR_VERSION
#endif
#include <pcap.h>


#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <signal.h>
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#include <dnet.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "personality.h"
#include "xprobe_assoc.h"
#include "ipfrag.h"
#include "router.h"
#include "tcp.h"
#include "udp.h"

/* Prototypes */
int pcap_dloff(pcap_t *);

void honeyd_tcp_timeout(int, short, void *);
void honeyd_udp_timeout(int, short, void *);
void tcp_retrans_timeout(int, short, void *);

SPLAY_HEAD(tree, tuple) tcpcons;
SPLAY_HEAD(utree, tuple) udpcons;

u_char echopkt[65536];

#define DIFF(a,b) do { \
	if ((a) < (b)) return -1; \
	if ((a) > (b)) return 1; \
} while (0)

int
compare(struct tuple *a, struct tuple *b)
{
	DIFF(a->ip_src, b->ip_src);
	DIFF(a->ip_dst, b->ip_dst);
	DIFF(a->sport, b->sport);
	DIFF(a->dport, b->dport);

	return (0);
}

SPLAY_PROTOTYPE(tree, tuple, node, compare);
SPLAY_GENERATE(tree, tuple, node, compare);

SPLAY_PROTOTYPE(utree, tuple, node, compare);
SPLAY_GENERATE(utree, tuple, node, compare);

pcap_t			*honeyd_pcap;
arp_t			*honeyd_arp;
ip_t			*honeyd_ip;
rand_t			*honeyd_rand;
struct intf_entry	 honeyd_ifent;
int			 honeyd_sig;
int			 honeyd_dloff;
int			 honeyd_nconnects;
int			 honeyd_nchildren;
int			 honeyd_dopoll;
int			 honeyd_ttl;
struct tcp_con		 honeyd_tmp;
#ifdef WIN32
int debug;
#endif

void
usage(void)
{
	fprintf(stderr,
#ifdef WIN32
			"WIN32 Port By Michael A. Davis (mdavis@securityprofiling.com, www.securityprofiling.com)\n"
			"Usage: honeyd [-dPW] [-l logfile] [-i interface] [-p personalities]\n"
#else
	    "Usage: honeyd [-dP] [-l logfile] [-i interface] [-p personalities]\n"
#endif
	    "\t[-x xprobe] [-a assoc] [-f config] [net ...]\n");
	exit(1);
}

void
honeyd_settcp(struct tcp_con *con, struct ip_hdr *ip, struct tcp_hdr *tcp,
    int local)
{
	struct tuple *hdr = &con->conhdr;

	memset(hdr, 0, sizeof(struct tuple));
	hdr->ip_src = ip->ip_src;
	hdr->ip_dst = ip->ip_dst;
	hdr->sport = ntohs(tcp->th_sport);
	hdr->dport = ntohs(tcp->th_dport);
	hdr->type = SOCK_STREAM;
	hdr->local = local;
	con->rcv_flags = tcp->th_flags;
	con->cmd.pfd = INVALID_SOCKET;
	con->cmd.perrfd = INVALID_SOCKET;
#ifdef WIN32
	con->cmd.pwritefd = INVALID_SOCKET;
	con->cmd.IsProxyCmd = 0;
#endif
}

void
honeyd_setudp(struct udp_con *con, struct ip_hdr *ip, struct udp_hdr *udp,
    int local)
{
	struct tuple *hdr = &con->conhdr;

	memset(hdr, 0, sizeof(struct tuple));
	hdr->ip_src = ip->ip_src;
	hdr->ip_dst = ip->ip_dst;
	hdr->sport = ntohs(udp->uh_sport);
	hdr->dport = ntohs(udp->uh_dport);
	hdr->type = SOCK_DGRAM;
	hdr->local = local;
	con->softerrors = 0;
	con->cmd.pfd = INVALID_SOCKET;
	con->cmd.perrfd = INVALID_SOCKET;
#ifdef WIN32
	con->cmd.pwritefd = INVALID_SOCKET;
	con->cmd.IsProxyCmd = 0;
#endif
	TAILQ_INIT(&con->incoming);
}

char *
honeyd_contoa(struct tuple *hdr)
{
	static char buf[128];
	char asrc[24], adst[24];
	struct addr src, dst;
	u_short sport, dport;
	
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_src, IP_ADDR_LEN);
	addr_pack(&dst, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_dst, IP_ADDR_LEN);

	/* For a local connection switch the address around */
	if (hdr->local) {
		struct addr tmp;

		tmp = src;
		src = dst;
		dst = tmp;

		sport = hdr->dport;
		dport = hdr->sport;
	} else {
		sport = hdr->sport;
		dport = hdr->dport;
	}

	addr_ntop(&src, asrc, sizeof(asrc));
	addr_ntop(&dst, adst, sizeof(adst));

	snprintf(buf, sizeof(buf), "(%s:%d - %s:%d)",
	    asrc, sport, adst, dport);

	return (buf);
}

/*
 * Expands several command line arguments into a complete pcap filter string.
 * Deals with normal CIDR notation and IP-IP ranges.
 */

static char *
honeyd_expandips(int naddresses, char **addresses)
{
	static char filter[1024];
	char line[1024], *p;
	struct addr dst;

	if (naddresses == 0)
		return (NULL);

	filter[0] = '\0';

	while (naddresses--) {
		/* Get current address */
		p = *addresses++;

		if (filter[0] != '\0') {
			if (strlcat(filter, " or ", sizeof(filter)) >= sizeof(filter))
				errx(1, "%s: too many address for filter", 
				    __func__);
		}

		if (addr_pton(p, &dst) != -1) {

			snprintf(line, sizeof(line), "dst %s%s",
			    dst.addr_bits != 32 ? "net ": "", p);
		} else {
			char *first, *second;
			struct addr astart, aend;
			struct in_addr in;
			ip_addr_t istart, iend;

			second = p;

			first = strsep(&second, "-");
			if (second == NULL)
				errx(1, "%s: Invalid network range: %s",
				    __func__, p);

			line[0] = '\0';
			if (addr_pton(first, &astart) == -1 ||
			    addr_pton(second, &aend) == -1)
				errx(1, "%s: bad addresses %s-%s", __func__,
				    first, second);
#ifdef HAVE_BROKEN_DNET
			if (addr_cmp(&astart, &aend) <= 0)
#else
			if (addr_cmp(&astart, &aend) >= 0)
#endif
			    errx(1, "%s: inverted range %s-%s", __func__,
				first, second);

			/* Completely, IPv4 specific */
			istart = ntohl(astart.addr_ip);
			iend = ntohl(aend.addr_ip);
			while (istart <= iend) {
				char single[32];
				int count = 0, done = 0;
				ip_addr_t tmp;

				do {
					ip_addr_t bit = 1 << count;
					ip_addr_t mask;

					mask = ~(~0 << count);
					tmp = istart | mask;

					if (istart & bit)
						done = 1;

					if (iend < tmp) {
						count--;
						mask = ~(~0 << count);
						tmp = istart | mask;
						break;
					} else if (done)
						break;
					
					count++;
				} while (count < IP_ADDR_BITS);

				if (line[0] != '\0')
					strlcat(line, " or ", sizeof(line));
				in.s_addr = htonl(istart);
				snprintf(single, sizeof(single),
				    "dst net %s/%d",
				    inet_ntoa(in), 32 - count);

				strlcat(line, single, sizeof(line));

				istart = tmp + 1;
			}
		}
		
		if (strlcat(filter, line, sizeof(filter)) >= sizeof(filter))
			errx(1, "%s: too many address for filter", 
			    __func__);
	}

	return (filter);
}

void
honeyd_init(char *dev, int naddresses, char **addresses)
{
	struct bpf_program fcode;
#ifndef WIN32
	struct rlimit rl;
#else
	pcap_if_t *alldevs, *d;
	int adapter, i;
	LPADAPTER lpAdapter;
	NetType netType;
	PACKET_OID_DATA oidMac;
#endif
	char filter[1024], ebuf[PCAP_ERRBUF_SIZE], *dst;
	intf_t *intf;
	int options, time;

	/* Compute the monitored IP addresses */
	dst = honeyd_expandips(naddresses, addresses);

	/* Initalize ongoing connection state */
	SPLAY_INIT(&tcpcons);
	SPLAY_INIT(&udpcons);

	memset(&honeyd_tmp, 0, sizeof(honeyd_tmp));

#ifndef WIN32
	/* Raising file descriptor limits */
	if (getrlimit(RLIMIT_NOFILE, &rl) == -1)
		err(1, "getrlimit");
	rl.rlim_cur=rl.rlim_max;
	if (setrlimit(RLIMIT_NOFILE, &rl) == -1)
		err(1, "setrlimit");
#ifdef RLIMIT_NPROC
	if (getrlimit(RLIMIT_NPROC, &rl) == -1)
		err(1, "getrlimit");
	rl.rlim_cur=rl.rlim_max;
	if (setrlimit(RLIMIT_NPROC, &rl) == -1)
		err(1, "setrlimit");
#endif
#endif

	if ((honeyd_ip = ip_open()) == NULL)
		err(1, "ip_open");

	if ((honeyd_arp = arp_open()) == NULL)
		err(1, "arp_open");

	if ((intf = intf_open()) == NULL)
		err(1, "intf_open");

#ifdef WIN32
	if (pcap_findalldevs(&alldevs, ebuf) == -1)
		errx(1, "pcap_findalldevs: %s", ebuf);
	
	d = alldevs;
	if (dev != NULL) {
		adapter = atoi(dev);
		// There should never be an adapter 0
		if(adapter == 0)
			errx(1, "Unable to find adapter %d", adapter);

		i = adapter;
		for(i; i > 1 && d != NULL; i--, d=d->next);
	
		if (d == NULL)
			errx(1, "Unable to find adapter %d", adapter);
	}
	dev = d->name;
	honeyd_ifent.intf_len = sizeof(honeyd_ifent);
	strlcpy(honeyd_ifent.intf_name, d->description, sizeof(honeyd_ifent.intf_name));
	
	lpAdapter = PacketOpenAdapter(dev);

	if (!lpAdapter || (lpAdapter->hFile == INVALID_HANDLE_VALUE))
		errx(1, "%s: Unable to open adapter %s. Error Code: %d", __func__, dev, GetLastError());

	PacketGetNetType(lpAdapter, &netType);
	if(netType.LinkType == NdisMedium802_3) {
		honeyd_ifent.intf_link_addr.addr_type = ADDR_TYPE_ETH;
		honeyd_ifent.intf_link_addr.addr_bits = ETH_ADDR_BITS;

		// Get the MAC Address and store it into the libdnet structure.
		oidMac.Oid = OID_802_3_PERMANENT_ADDRESS;
		oidMac.Length = ETH_ADDR_LEN;

		if(PacketRequest(lpAdapter, 0, &oidMac) == 0)
			errx(1, "%s: Unable to query adapter %s for MAC Address. Error Code: %d", __func__, dev, GetLastError());

		memcpy(&honeyd_ifent.intf_link_addr.addr_eth, oidMac.Data, ETH_ADDR_LEN);

		// Set the IP address associated with this NIC.
		honeyd_ifent.intf_addr.addr_type = ADDR_TYPE_IP;
		honeyd_ifent.intf_addr.addr_ip = ((struct sockaddr_in *)d->addresses->addr)->sin_addr.s_addr;
		honeyd_ifent.intf_addr.addr_bits = IP_ADDR_BITS;
	}

	PacketCloseAdapter(lpAdapter);
#else
	if (dev == NULL) {
		if ((dev = pcap_lookupdev(ebuf)) == NULL)
			errx(1, "pcap_lookupdev: %s", ebuf);
	}
	honeyd_ifent.intf_len = sizeof(honeyd_ifent);
	strlcpy(honeyd_ifent.intf_name, dev, sizeof(honeyd_ifent.intf_name));

	if (intf_get(intf, &honeyd_ifent) < 0)
		err(1, "intf_get");
#endif

	if (honeyd_ifent.intf_addr.addr_type != ADDR_TYPE_IP)
		errx(1, "bad interface configuration: not IP");

	if (snprintf(filter, sizeof(filter), "ip %s%s%s",
		dst ? "and (" : "", dst ? dst : "", dst ? ")" : "") >= sizeof(filter))
		errx(1, "%s: pcap filter exceeds maximum length", __func__);

	/* If we are ethernet, ignore our own packets */
	if (honeyd_ifent.intf_link_addr.addr_type == ADDR_TYPE_ETH) {
		char line[48];
		snprintf(line, sizeof(line), " and not ether src %s",
		    addr_ntoa(&honeyd_ifent.intf_link_addr));
		strlcat(filter, line, sizeof(filter));
	}

	time = honeyd_dopoll ? 10 : 100;
	if ((honeyd_pcap = pcap_open_live(dev, 1540, 0, time, ebuf)) == NULL)
		errx(1, "pcap_open_live: %s", ebuf);

	/* Get offset to packet data */
	honeyd_dloff = pcap_dloff(honeyd_pcap);
	
	if (pcap_compile(honeyd_pcap, &fcode, filter, 1, 0) < 0 ||
	    pcap_setfilter(honeyd_pcap, &fcode) < 0)
		errx(1, "bad pcap filter: %s", pcap_geterr(honeyd_pcap));
#if defined(BSD) && defined(BIOCIMMEDIATE)
	{
		int on = 1;
		if (ioctl(pcap_fileno(honeyd_pcap), BIOCIMMEDIATE, &on) < 0)
			warn("BIOCIMMEDIATE");
	}
#endif
#ifdef LOG_PERROR
	options = LOG_PERROR|LOG_PID|LOG_CONS;
#else
	options = LOG_PID|LOG_CONS;
#endif
	openlog("honeyd", options, LOG_DAEMON);
	syslog(LOG_INFO, "listening on %s: %s", dev, filter);
}

void
honeyd_exit(int status)
{
	honeyd_logend();

	pcap_close(honeyd_pcap);
	rand_close(honeyd_rand);
	arp_close(honeyd_arp);
	ip_close(honeyd_ip);
	closelog();
	unlink(PIDFILE);

	exit(status);
}

struct tcp_con *
tcp_new(struct ip_hdr *ip, struct tcp_hdr *tcp, int local)
{
	struct tcp_con *con;

	if ((con = calloc(1, sizeof(struct tcp_con))) == NULL) {
			syslog(LOG_WARNING, "calloc: %m");
			return (NULL);
	}
#ifdef WIN32
	con->cmd.IsProxyCmd = 0;
#endif
	honeyd_nconnects++;
	honeyd_settcp(con, ip, tcp, local);
	timeout_set(&con->timeout, honeyd_tcp_timeout, con);
	timeout_set(&con->retrans_timeout, tcp_retrans_timeout, con);

	SPLAY_INSERT(tree, &tcpcons, &con->conhdr);

	honeyd_log_flownew(IP_PROTO_TCP, &con->conhdr);
	return (con);
}

void
tcp_free(struct tcp_con *con)
{
	struct port *port = con->port;

	if (port != NULL)
		port_free(port->sub->tmpl, port);

	SPLAY_REMOVE(tree, &tcpcons, &con->conhdr);

	honeyd_log_flowend(IP_PROTO_TCP, &con->conhdr);

	timeout_del(&con->timeout);
	timeout_del(&con->retrans_timeout);

	if (con->cmd_pfd > 0)
		cmd_free(&con->cmd);

	honeyd_nconnects--;
	free(con);
}

void
tcp_retrans_timeout(int fd, short event, void *arg)
{
	struct tcp_con *con = arg;

	con->poff = 0;
	con->retrans_time *= 2;

	if (con->state == TCP_STATE_SYN_SENT) {
		if (con->retrans_time >= 60) {
			tcp_free(con);
			return;
		}
		con->snd_una--;
		tcp_send(con, TH_SYN, NULL, 0);
		con->snd_una++;

		generic_timeout(&con->retrans_timeout, con->retrans_time);
	} else
		tcp_senddata(con, TH_ACK);
}

struct udp_con *
udp_new(struct ip_hdr *ip, struct udp_hdr *udp, int local)
{
	struct udp_con *con;

	if ((con = calloc(1, sizeof(struct udp_con))) == NULL) {
			syslog(LOG_WARNING, "calloc: %m");
			return (NULL);
	}

	honeyd_setudp(con, ip, udp, local);

	SPLAY_INSERT(utree, &udpcons, &con->conhdr);

	timeout_set(&con->timeout, honeyd_udp_timeout, con);

	honeyd_log_flownew(IP_PROTO_UDP, &con->conhdr);

	return (con);
}

void
udp_free(struct udp_con *con)
{
	struct buffer *buf;
	struct port *port = con->port;

	if (port != NULL)
		port_free(port->sub->tmpl, port);

	SPLAY_REMOVE(utree, &udpcons, &con->conhdr);

	honeyd_log_flowend(IP_PROTO_UDP, &con->conhdr);

	while ((buf = TAILQ_FIRST(&con->incoming)) != NULL) {
		TAILQ_REMOVE(&con->incoming, buf, next);
		free(buf->buf);
		free(buf);
	}

	if (con->cmd_pfd > 0)
		cmd_free(&con->cmd);

	timeout_del(&con->timeout);
	free(con);
}

void
honeyd_tcp_timeout(int fd, short event, void *arg)
{
	struct tcp_con *con = arg;

	syslog(LOG_DEBUG, "Expiring TCP %s (%p) in state %d",
	    honeyd_contoa(&con->conhdr), con, con->state);

	tcp_free(con);
}

void
honeyd_udp_timeout(int fd, short event, void *arg)
{
	struct udp_con *con = arg;

	syslog(LOG_DEBUG, "Expiring UDP %s (%p)",
	    honeyd_contoa(&con->conhdr), con);

	udp_free(con);
}

struct action *
honeyd_protocol(struct template *tmpl, int proto)
{
	switch (proto) {
	case IP_PROTO_TCP:
		return (&tmpl->tcp);
	case IP_PROTO_UDP:
		return (&tmpl->udp);
	case IP_PROTO_ICMP:
		return (&tmpl->icmp);
	default:
		return (NULL);
	}
}

int
honeyd_block(struct template *tmpl, int proto, int number)
{
	struct port *port;
	struct action *action;

	if (tmpl == NULL)
		return (0);

	port = port_find(tmpl, proto, number);
	if (port == NULL)
		action = honeyd_protocol(tmpl, proto);
	else
		action = &port->action;

	return (action->status == PORT_BLOCK);
}

void
honeyd_varexpand(struct tcp_con *con, char *line, u_int linesize)
{
	char asc[32], *p;
	struct addr src, dst;

	addr_pack(&dst, ADDR_TYPE_IP, IP_ADDR_BITS, &con->con_ipdst, IP_ADDR_LEN);
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &con->con_ipsrc, IP_ADDR_LEN);

	/* Do some simple replacements */
	p = addr_ntoa(&src);
        while (strrpl(line, linesize, "$ipsrc", p) != NULL)
                ;
	p = addr_ntoa(&dst);
        while (strrpl(line, linesize, "$ipdst", p) != NULL)
                ;
	snprintf(asc, sizeof(asc), "%d", con->con_sport);
        while (strrpl(line, linesize, "$sport", asc) != NULL)
                ;
	snprintf(asc, sizeof(asc), "%d", con->con_dport);
        while (strrpl(line, linesize, "$dport", asc) != NULL)
                ;
}

/*
 * Determines if a particular port is open or closed based on looking
 * at the default template of connections.
 */

int
honeyd_portopen(struct template *tmpl, int proto, u_short number)
{
	struct port *port;
	struct action *action;
	
	if (tmpl == NULL) {
		switch (proto) {
		case IP_PROTO_UDP:
			return (0);
		default:
			return (1);  /* Default */
		}
	}

	port = port_find(tmpl, proto, number);
	if (port == NULL)
		action = honeyd_protocol(tmpl, proto);
	else
		action = &port->action;

	return (PORT_ISOPEN(action));
}

/* 
 * Create a proxy connection, either use precomputed addrinfo or
 * generate correct address information.
 */

int
proxy_connect(struct tuple *hdr, struct command *cmd, struct addrinfo *ai,
    char *line, void *arg)
{
	int res;

	/* Check if the address has been resolved for us already */
	if (ai == NULL) {
		char *name, *strport = line;
		u_short nport;

		name = strsep(&strport, ":");
		if (strport == NULL || (nport = atoi(strport)) == 0)
			return (-1);

		if ((ai = cmd_proxy_getinfo(name, hdr->type, nport)) == NULL)
			return (-1);
		res = cmd_proxy_connect(hdr, cmd, ai, arg);
#ifdef WIN32
		if(ai->ai_addr)
			free(ai->ai_addr);
		free(ai);
#else
		freeaddrinfo(ai);
#endif
	} else
		res = cmd_proxy_connect(hdr, cmd, ai, arg);

	return (res);
}

/* Cleans up receive and send buffers if cmd does not start */

void
tcp_connectfail(struct tcp_con *con)
{
	if (con->payload) {
		free(con->payload);
		con->payload = NULL;
	}
	if (con->readbuf) {
		free(con->readbuf);
		con->readbuf = NULL;
	}
}

/* Sets up buffers for a fully connected TCP connection */

int
tcp_setupconnect(struct tcp_con *con)
{
	struct tuple *hdr = &con->conhdr;

	/* Allocate buffers */
	if ((con->payload = malloc(TCP_DEFAULT_SIZE)) == NULL) {
		syslog(LOG_WARNING, "malloc %s: %m", honeyd_contoa(hdr));
		goto err;
	}
	if ((con->readbuf = malloc(TCP_DEFAULT_SIZE)) == NULL) {
		syslog(LOG_WARNING, "malloc %s: %m", honeyd_contoa(hdr));
		goto err;
	}
	con->psize = TCP_DEFAULT_SIZE;
	con->rsize = TCP_DEFAULT_SIZE;

	return (0);

 err:
	tcp_connectfail(con);

	return (-1);
}

void
generic_connect(struct tuple *hdr, struct command *cmd, void *con)
{
	char line[1024];
	char *argv[4];
	struct addr dst;
	struct template *tmpl;
	struct action *action = NULL;
	struct port *port;
	int proto = 0;

	if (hdr->type == SOCK_STREAM)
		proto = IP_PROTO_TCP;
	else
		proto = IP_PROTO_UDP;
	
	addr_pack(&dst, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_dst, IP_ADDR_LEN);
	tmpl = template_find(addr_ntoa(&dst));
	if (tmpl == NULL)
		tmpl = template_find("default");
	if (tmpl == NULL)
		goto out;

	if ((port = port_find(tmpl, proto, hdr->dport)) == NULL) {
		action = &tmpl->tcp;
	} else
		action = &port->action;

	if (action->status == PORT_OPEN) {
		if (action->action == NULL || strlen(action->action) == 0)
			goto out;
	}

	if (proto == IP_PROTO_TCP && tcp_setupconnect(con) == -1)
		goto out;

	/* Connect to the already started sub system */
	if (action->status == PORT_SUBSYSTEM) {
		if (cmd_subsystem_connect(hdr, cmd, port, con) == -1)
			goto out;
		return;
	}

	/* 3-way handshake has been completed */
	if (proto == IP_PROTO_TCP && action->status == PORT_RESERVED) {
		if (cmd_subsystem_localconnect(hdr, cmd, port, con) == -1)
			goto err;
		return;
	}

	if (action->status == PORT_OPEN || action->aitop == NULL) {
		strlcpy(line, action->action, sizeof(line));
		honeyd_varexpand(con, line, sizeof(line));
	}

	/* Setup a proxy connection, no need to fork a new process */
	if (action->status == PORT_PROXY) {
		int res;

		res = proxy_connect(hdr, cmd, action->aitop, line, con);
		if (res == -1)
			goto out;
		return;
	}
#ifdef WIN32
	argv[0] = "C:\\winnt\\system32\\cmd.exe";
	argv[1] = "/c";
	argv[2] = line;
	argv[3] = NULL;

	if (cmd_fork(hdr, cmd, tmpl, "C:\\winnt\\system32\\cmd.exe", argv, con) == -1) {
#else
	argv[0] = "/bin/sh";
	argv[1] = "-c";
	argv[2] = line;
	argv[3] = NULL;

	if (cmd_fork(hdr, cmd, tmpl, "/bin/sh", argv, con) == -1) {
#endif

		syslog(LOG_WARNING, "malloc %s: %m", honeyd_contoa(hdr));
		goto err;
	}

	syslog(LOG_INFO, "Connection established: %s %s <-> %s",
	    proto == IP_PROTO_TCP ? "tcp" : "udp",
	    honeyd_contoa(hdr), line);
	return;

 err:
	if (proto == IP_PROTO_TCP)
		tcp_connectfail(con);
 out:
	syslog(LOG_INFO, "Connection established: %s %s",
	    proto == IP_PROTO_TCP ? "tcp" : "udp",
	    honeyd_contoa(hdr));
}

int
tcp_send(struct tcp_con *con, uint8_t flags, u_char *payload, u_int len)
{
	u_char pkt[1500];
	struct tcp_hdr *tcp;
	u_int iplen;
	int window = 16000;
	int dontfragment = 0;
	char *options;
	uint16_t id = rand_uint16(honeyd_rand);

	if (con->window)
		window = con->window;

	options = tcp_personality(con, &flags, &window, &dontfragment, &id);

	/* Empty flags indicates packet drop */
	if (flags == 0)
		return (0);

	if ((flags & TH_SYN) && !con->window)
		con->window = window;

	/* Simple window tracking */
	if (window && con->rlen) {
		window -= con->rlen;
		if (window < 0)
			window = 0;
	}

	/* Set sequence number */
	if (con->snd_una == 0)
		con->snd_una = rand_uint32(honeyd_rand);

	tcp = (struct tcp_hdr *)(pkt + IP_HDR_LEN);
	tcp_pack_hdr(tcp,
	    con->con_dport, con->con_sport,
	    con->snd_una, con->rcv_next, flags, window, 0);

	/* ET - options is non-NULL if a personality was found.  If a
         * personality was found, it means that this packet is a response
         * to an NMAP TCP test (not a Sequence number test, a weird flags test).
 	 * Therefore if options is not NULL, you have to add the options to
         * the response otherwise the reply packet will not have the complete
         * personality.  Of the seven NMAP TCP tests, only a couple may
         * return a packet with the SYN flag.  I needed to remove the
         * requirement of the SYN flag so that the other NMAP TCP tests would
         * have the personality TCP options. */

	if (options != NULL)
		tcp_personality_options(con, tcp, options);

	iplen = IP_HDR_LEN + (tcp->th_off << 2) + len;
	
	/* Src and Dst are reversed both for ip and tcp */
	ip_pack_hdr(pkt, 0, iplen, id,
	    dontfragment ? IP_DF : 0, honeyd_ttl,
	    IP_PROTO_TCP, con->con_ipdst, con->con_ipsrc);

	memcpy(pkt + IP_HDR_LEN + (tcp->th_off << 2), payload, len);

	ip_checksum(pkt, iplen);
	
	if (ip_send(honeyd_ip, pkt, iplen) != iplen)
		syslog(LOG_ERR, "couldn't send packet: %m");

	return (len);
}

void
tcp_senddata(struct tcp_con *con, uint8_t flags)
{
	unsigned int space, sent;
	int needretrans = 0;

	do {
		space = TCP_MAX_INFLIGHT - TCP_BYTESINFLIGHT(con);
		if (space > TCP_MAX_SEND)
			space = TCP_MAX_SEND;
		if (con->plen - con->poff < space)
			space = con->plen - con->poff;

		if (con->sentfin && !con->finacked)
			flags |= TH_FIN;
		if (con->plen > space)
			flags &= ~TH_FIN;

		/*
		 * If we do not ack new data, and have nothing to send,
		 * and do not need to send a FIN, stop sending.
		 */
		if (space == 0 && con->last_acked == con->rcv_next &&
		    !(flags & TH_FIN))
			break;

		con->snd_una += con->poff;
		sent = tcp_send(con, flags, con->payload + con->poff, space);
		con->snd_una -= con->poff;
		con->poff += sent;

		/* Statistics */
		con->conhdr.sent += space;

		if (flags & TH_ACK)
			con->last_acked = con->rcv_next;

	} while (sent && !con->dupacks);

	/* 
	 * We need to retransmit if we still have outstanding data or
	 * our FIN did not get acked.
	 */
	needretrans = con->poff || (con->sentfin && !con->finacked);

	if (needretrans && !timeout_pending(&con->retrans_timeout, NULL)) {
		if (!con->retrans_time)
			con->retrans_time = 1;
		generic_timeout(&con->retrans_timeout, con->retrans_time);
	}
}

void
tcp_sendfin(struct tcp_con *con)
{
	con->sentfin = 1;
	tcp_senddata(con, TH_ACK);
	switch (con->state) {
	case TCP_STATE_ESTABLISHED:
		con->state = TCP_STATE_FIN_WAIT_1;
		break;
	case TCP_STATE_CLOSE_WAIT:
		con->state = TCP_STATE_CLOSING;
		break;
	}
}

void
icmp_error_send(struct addr *addr, uint8_t type, uint8_t code,
    struct ip_hdr *rip)
{
	u_char pkt[1500];
	u_int iplen;
	uint16_t ipid;
	uint8_t tos = 0, df = 0, ttl = honeyd_ttl;
	ip_addr_t src;
	int quotelen, riplen;

	quotelen = 40;

	if (!icmp_error_personality(addr, rip, &df, &tos, &quotelen, &ttl))
		return;

	memcpy(&src, addr->addr_data8, sizeof(src));

	riplen = ntohs(rip->ip_len);
	if (riplen < quotelen)
		quotelen = riplen;

	iplen = IP_HDR_LEN + ICMP_HDR_LEN + 4 + quotelen;
	ipid = rand_uint16(honeyd_rand);

	ip_pack_hdr(pkt, tos, iplen, ipid, df ? IP_DF: 0, ttl,
	    IP_PROTO_ICMP, src, rip->ip_src);
	icmp_pack_hdr_quote(pkt + IP_HDR_LEN, type, code, 0, rip, quotelen);

	ip_checksum(pkt, iplen);
	
	if (ip_send(honeyd_ip, pkt, iplen) != iplen)
		syslog(LOG_ERR, "couldn't send packet: %m");
}

/*
 * icmp_echo_reply
 * rip should be the ip_header pointing to an actual raw
 * packet (has payload in it so icmp can be extracted)
 *
 * This function changes the IP and ICMP header data (i.e.
 * the ICMP packet and its IP header) to match the OS you want.
 *
 * The code, ipid, tos, offset, and ttl parameters should 
 * probably move inside this function so that the icmp_personality 
 * can have control over them.
 * 
 * We should create a structure that includes all possible ICMP
 * fields and just pass in that structure into icmp_personality
 * function and have a flag to indicate what ICMP message it is
 * and what parameters need to set to what value depend on the
 * OS we need to emulate.
 * e.g.
	if (!icmp_personality(addr, rip, &ICMP_STRUCT)
		return;
 *
 * param rip The raw ip packet in IP header form
 * param code ICMP Header REPLY echo code, OS dependent, 0 or !0
 * param ipid IP Header id, 0 or !0(RANDOM)
 * param tos IP Header type of service, 0 or !0(0xc0)
 * param offset IP Header DF bit and offset, 0 or 1(IP_DF)
 * param ttl IP header time to live, <65, <129, or <256
 * param payload ICMP Echo request payload, should not be null, use by
 * 		ping programs to determine RTT
 * param len Length of the payload to return
 */
void
icmp_echo_reply(struct ip_hdr *rip, uint8_t code, uint16_t ipid,
    uint8_t tos, uint16_t offset, uint8_t ttl,
    u_char *payload, u_int len)
{
	u_char pkt[1500];
	u_int iplen;
	struct icmp_msg_echo *icmp_echo;
       
	icmp_echo = (struct icmp_msg_echo *) ((u_char *)rip + (rip->ip_hl << 2) + ICMP_HDR_LEN);

	iplen = IP_HDR_LEN + ICMP_HDR_LEN + 4 + len;

	ip_pack_hdr(pkt, tos, iplen, ipid, offset, ttl,
	    IP_PROTO_ICMP, rip->ip_dst, rip->ip_src);
	icmp_pack_hdr_echo(pkt + IP_HDR_LEN, ICMP_ECHOREPLY, 
		code, ntohs(icmp_echo->icmp_id), ntohs(icmp_echo->icmp_seq), 
		payload, len);

	ip_checksum(pkt, iplen);

	if (ip_send(honeyd_ip, pkt, iplen) != iplen)
	syslog(LOG_ERR, "couldn't send packet: %m");
}

/*
 * We should create a structure that includes all possible ICMP
 * fields and just pass in that structure into icmp_personality
 * function and have a flag to indicate what ICMP message it is
 * and what parameters need to set to what value depend on the
 * OS we need to emulate.
 * e.g.
	if (!icmp_personality(addr, rip, &ICMP_STRUCT)
		return;
		
 * ICMP_TIMESTAMP REPLY, 
 *
 * param rip The raw ip packet in IP header form
 * param icmp_rip icmp timestamp message, includes the
 * 	icmp header.
 * param ttl Time to live of the emulated OS (<65, <129, <256)
 */ 
void
icmp_timestamp_reply(struct ip_hdr *rip,
    struct icmp_msg_timestamp* icmp_rip, uint8_t ttl)
{
	u_char pkt[1500];
	u_int iplen;
	uint16_t ipid;
	struct icmp_msg_timestamp icmp_time;
	uint8_t padding = 6;
	struct tm *now_tm;
	time_t now;
	uint32_t milliseconds;

	now = time(NULL);
	now_tm = localtime(&now);

	milliseconds = (now_tm->tm_hour * 60 * 60 + 
		now_tm->tm_min * 60 + 
		now_tm->tm_sec) * 1000;

	icmp_time.hdr.icmp_type = ICMP_TSTAMPREPLY,
	icmp_time.hdr.icmp_code = 0;
	icmp_time.icmp_id = icmp_rip->icmp_id;
	icmp_time.icmp_seq = icmp_rip->icmp_seq;

	/* For now just do the following */
	icmp_time.icmp_ts_orig = icmp_rip->icmp_ts_orig;
	icmp_time.icmp_ts_rx = icmp_rip->icmp_ts_orig + milliseconds;
	icmp_time.icmp_ts_tx = icmp_rip->icmp_ts_rx;
		
	iplen = IP_HDR_LEN + ICMP_HDR_LEN + 16 + padding; 
	/* 6 bytes of 0 at the end, why? RedHat and Windows have 6 bytes of
	 * padding to this type of message. I don't know yet why they do this */
	ipid = rand_uint16(honeyd_rand);
	
	ip_pack_hdr(pkt, rip->ip_tos, iplen, ipid, rip->ip_off, ttl,
	    IP_PROTO_ICMP, rip->ip_dst,rip->ip_src);
	memcpy(pkt + IP_HDR_LEN, &icmp_time, sizeof(icmp_time));
			
	ip_checksum(pkt, iplen);
	
	if (ip_send(honeyd_ip, pkt, iplen) != iplen)
		syslog(LOG_ERR, "couldn't send packet: %m"); 
}

/*
 * ICMP_MASK_REPLY.

 * We should create a structure that includes all possible ICMP
 * fields and just pass in that structure into icmp_personality
 * function and have a flag to indicate what ICMP message it is
 * and what parameters need to set to what value depend on the
 * OS we need to emulate.
 * e.g.
	if (!icmp_personality(addr, rip, &ICMP_STRUCT)
		return;
	
 * param rip The raw ip packet in IP header form
 * param idseq id and seq of the icmp header, should be same
 * 	from the mask request icmp header.
 * param ttl time to live of OS simulated (<65, <129, or <255)
 * param mask mask of the emulated OS (i.e. 255.255.0.0)
 */ 
void
icmp_mask_reply(struct ip_hdr *rip, 
	struct icmp_msg_idseq *idseq, uint8_t ttl, uint32_t addrmask)
{
	u_char pkt[1500];
	u_int iplen;
	uint16_t ipid;
	struct icmp_mesg_mask mask;
	
	iplen = IP_HDR_LEN + ICMP_HDR_LEN + 8; 
	ipid = rand_uint16(honeyd_rand);

	mask.hdr.icmp_type = ICMP_MASKREPLY;
	mask.hdr.icmp_code = ICMP_CODE_NONE;
	
	mask.icmp_id = idseq->icmp_id;
	mask.icmp_seq = idseq->icmp_seq;
	mask.icmp_mask = htonl(addrmask);
	
	ip_pack_hdr(pkt, rip->ip_tos, iplen, ipid, rip->ip_off, ttl,
	    IP_PROTO_ICMP, rip->ip_dst,rip->ip_src);
	memcpy(pkt + IP_HDR_LEN, &mask, sizeof(mask));
			
	ip_checksum(pkt, iplen);
	
	if (ip_send(honeyd_ip, pkt, iplen) != iplen)
		syslog(LOG_ERR, "couldn't send packet: %m");
}

/*
 * We should create a structure that includes all possible ICMP
 * fields and just pass in that structure into icmp_personality
 * function and have a flag to indicate what ICMP message it is
 * and what parameters need to set to what value depend on the
 * OS we need to emulate.
 * e.g.
	if (!icmp_personality(addr, rip, &ICMP_STRUCT)
		return;
		
 * ICMP_INFO_REPLY
 *
 * param rip The raw ip packet in IP header form
 * param idseq id and seq of the icmp header, should be same
 * 	from the info request icmp header.
 * param ttl Time to live of the emulated OS (<65, <129, <256)
 */ 
void
icmp_info_reply(struct ip_hdr *rip, 
		struct icmp_msg_idseq *idseq, uint8_t ttl)
{
	u_char pkt[1500];
	u_int iplen;
	uint16_t ipid;
	struct icmp_msg_inforeply inforeply;
	
	iplen = IP_HDR_LEN + ICMP_HDR_LEN + 4; 
	ipid = rand_uint16(honeyd_rand);

	inforeply.hdr.icmp_type = ICMP_INFOREPLY;
	inforeply.hdr.icmp_code = ICMP_CODE_NONE;
	
	inforeply.idseq.icmp_id = idseq->icmp_id;
	inforeply.idseq.icmp_seq = idseq->icmp_seq;
	
	ip_pack_hdr(pkt, rip->ip_tos, iplen, ipid, rip->ip_off, ttl,
	    IP_PROTO_ICMP, rip->ip_dst,rip->ip_src);
	memcpy(pkt + IP_HDR_LEN, &inforeply, sizeof(inforeply));
			
	ip_checksum(pkt, iplen);
	
	if (ip_send(honeyd_ip, pkt, iplen) != iplen)
		syslog(LOG_ERR, "couldn't send packet: %m");
}

void
tcp_do_options(struct tcp_con *con, struct tcp_hdr *tcp)
{
	u_char *p, *end;

	p = (u_char *)(tcp + 1);
	end = (u_char *)tcp + (tcp->th_off << 2);

	while (p < end) {
		struct tcp_opt opt, *tmp = (struct tcp_opt *)p;

		if (tmp->opt_type == TCP_OPT_NOP) {
			p++;
			continue;
		} else if (tmp->opt_type == TCP_OPT_EOL)
			break;

		if (p + tmp->opt_len > end)
			break;

		memcpy(&opt, tmp, tmp->opt_len);
		switch (opt.opt_type) {
		case TCP_OPT_MSS:
			con->mss = ntohs(opt.opt_data.mss);
			break;
		case TCP_OPT_WSCALE:
			con->sawwscale = 1;
			break;
		case TCP_OPT_TIMESTAMP:
			con->sawtimestamp = 1;
			con->echotimestamp = opt.opt_data.timestamp[0];
			break;
		default:
			break;
		}

		p += opt.opt_len;
		if (opt.opt_len < 1)
			break;
	}
}

void
generic_timeout(struct event *ev, int seconds)
{
	struct timeval tv;

	timerclear(&tv);
	tv.tv_sec = seconds;
	timeout_add(ev, &tv);
}

/* Checks that the sequence number is where we expect it to be */
#define TCP_CHECK_SEQ_OR_ACK	do { \
		if (!(tcp->th_flags & TH_ACK)) \
			goto dropwithreset; \
\
		if (TCP_SEQ_LT(th_ack, con->snd_una)) { \
			if (tcp->th_flags & TH_RST) \
				goto drop; \
		}\
		if (th_seq != con->rcv_next) { \
			tcp_send(con, TH_ACK, NULL, 0); \
			return; \
		} \
		if (tcp->th_flags & TH_RST) \
			goto close; \
} while(0)

#define TCP_RECV_SEND_DATA	do { \
		/* New data */ \
		dlen = ntohs(ip->ip_len) - (ip->ip_hl * 4) -(tcp->th_off * 4);\
\
		con->conhdr.received += dlen; \
\
		if (con->plen || con->cmd_pfd > 0) { \
			int ackinc = 0; \
			dlen = tcp_add_readbuf(con, data, dlen); \
\
			acked = th_ack - con->snd_una; \
			if (acked > con->plen) { \
				if (con->sentfin && acked == con->plen + 1){ \
					con->finacked = 1; \
					ackinc = 1; \
				} \
				acked = con->plen; \
			} \
			tcp_drain_payload(con, acked); \
			acked += ackinc; \
			if (con->cmd_pfd == INVALID_SOCKET && con->plen <= TCP_MAX_SEND) \
				con->sentfin = 1; \
		} else if (con->sentfin) { \
			if (th_ack == con->snd_una + 1) { \
				acked = 1; \
				con->finacked = 1; \
			} \
		} \
		if (acked == 0 && con->poff) { \
			con->dupacks++; \
			if (con->dupacks >= 3) { \
				con->dupacks = 3; \
				con->poff = 0; \
			} \
		} else if (acked) { \
			con->retrans_time = 0; \
			timeout_del(&con->retrans_timeout); \
			con->dupacks=0; \
		} \
} while (0)

void
tcp_recv_cb(struct template *tmpl, u_char *pkt, u_short pktlen)
{
	struct ip_hdr *ip;
	struct tcp_hdr *tcp;
	struct tcp_con *con;
	uint32_t th_seq, th_ack;
	uint32_t acked = 0;
	uint16_t th_sum;
	u_char *data;
	u_int dlen;
	uint8_t tiflags;

	ip = (struct ip_hdr *)pkt;
	tcp = (struct tcp_hdr *)(pkt + (ip->ip_hl << 2));
	data = (u_char *)(pkt + (ip->ip_hl*4) + (tcp->th_off*4));
	
	if (pktlen < (ip->ip_hl << 2) + TCP_HDR_LEN)
		return;

	if (honeyd_block(tmpl, IP_PROTO_TCP, ntohs(tcp->th_dport)))
		goto justlog;

	/* Check the checksum the brutal way, until libdnet supports */
	th_sum = tcp->th_sum;
	ip_checksum(ip, pktlen);
	if (th_sum != tcp->th_sum)
		goto justlog;

	honeyd_settcp(&honeyd_tmp, ip, tcp, 0);
	honeyd_tmp.state = honeyd_portopen(tmpl, IP_PROTO_TCP,
	    ntohs(tcp->th_dport)) ? TCP_STATE_LISTEN : TCP_STATE_CLOSED;

	con = (struct tcp_con *)SPLAY_FIND(tree, &tcpcons, &honeyd_tmp.conhdr);
	if (con == NULL) {
		if (honeyd_tmp.state != TCP_STATE_LISTEN)
			goto kill;

		if ((tcp->th_flags & TH_SYN) == 0) 
			goto kill;

		if (tcp->th_flags & ~TH_SYN &
		    (TH_FIN|TH_RST|TH_PUSH|TH_ACK|TH_URG)) {
			int win = 16000, df = 0;
			uint8_t flags = TH_SYN|TH_ACK;
			uint16_t id;
			/* Check what the flags say */
			tcp_personality(&honeyd_tmp, &flags, &win, &df, &id);
			if (flags & TH_RST)
				goto kill;
		}

		syslog(LOG_INFO, "Connection request: tcp %s",
		    honeyd_contoa(&honeyd_tmp.conhdr));

		/* Out of memory is dealt with by killing the connection */
		if ((con = tcp_new(ip, tcp, 0)) == NULL) {
			goto kill;
		}

		tcp_do_options(con, tcp);

		con->tmpl = tmpl;
		con->rcv_next = ntohl(tcp->th_seq) + 1;
		con->snd_una = 0;

		con->state = TCP_STATE_LISTEN;
		tcp_send(con, TH_SYN|TH_ACK, NULL, 0);

		con->snd_una++;
		con->state = TCP_STATE_SYN_RECEIVED;

		generic_timeout(&con->timeout, HONEYD_SYN_WAIT);

		return;
	}

	th_seq = ntohl(tcp->th_seq);
	th_ack = ntohl(tcp->th_ack);

	con->rcv_flags = tiflags = tcp->th_flags;

	switch (con->state) {
	case TCP_STATE_SYN_SENT:
		if (tiflags & TH_RST)
			goto close;
		if (!(tiflags & TH_SYN))
			goto drop;
		if (!(tiflags & TH_ACK))
			goto drop;

		/* No simultaenous open allowed */
		if (th_ack != con->snd_una)
			goto dropwithreset;

		con->rcv_next = th_seq + 1;
		tcp_send(con, TH_ACK, NULL, 0);

		con->state = TCP_STATE_ESTABLISHED;
		generic_connect(&con->conhdr, &con->cmd, con);
		break;

	case TCP_STATE_SYN_RECEIVED:
		if (tiflags & TH_ACK) {
			if (tiflags & TH_SYN)
				goto dropwithreset;
			if (th_ack != con->snd_una)
				goto dropwithreset;
		}
		if (tiflags & TH_SYN) {
			if (th_seq != con->rcv_next - 1)
				goto dropwithreset;
			con->snd_una--;
			tcp_send(con, TH_SYN|TH_ACK, NULL,0);
			con->snd_una++;
			return;
		}

		if (tiflags & TH_RST)
			goto close;
		if (!(tiflags & TH_ACK))
			goto drop;

		generic_timeout(&con->timeout, HONEYD_MAX_ACTIVE);

		con->state = TCP_STATE_ESTABLISHED;
		generic_connect(&con->conhdr, &con->cmd, con);
		break;

	case TCP_STATE_ESTABLISHED:
		TCP_CHECK_SEQ_OR_ACK;

		TCP_RECV_SEND_DATA;
			
		if (tiflags & TH_FIN) {
			if (con->cmd_pfd > 0)
				shutdown(con->cmd_pfd, SHUT_WR);
			else
				con->sentfin = 1;
			con->state = TCP_STATE_CLOSE_WAIT;
			dlen++;
		}

		con->rcv_next += dlen;
		con->snd_una += acked;
		if (con->sentfin) {
			tcp_sendfin(con);
		} else
			tcp_senddata(con, TH_ACK);
		break;

	case TCP_STATE_CLOSE_WAIT:
		TCP_CHECK_SEQ_OR_ACK;

		TCP_RECV_SEND_DATA;

		if (dlen)
			goto dropwithreset;
		con->snd_una += acked;
		tcp_senddata(con, TH_ACK);
		if (con->sentfin)
			con->state = TCP_STATE_CLOSING;

		break;

	case TCP_STATE_CLOSING:
		TCP_CHECK_SEQ_OR_ACK;

		TCP_RECV_SEND_DATA;

		con->snd_una += acked;
		if (con->finacked)
			goto closed;
		tcp_senddata(con, TH_ACK);
		break;

	case TCP_STATE_FIN_WAIT_1:
		TCP_CHECK_SEQ_OR_ACK;

		TCP_RECV_SEND_DATA;
			
		if (tiflags & TH_FIN) {
			con->state = TCP_STATE_CLOSING;
			generic_timeout(&con->timeout, HONEYD_CLOSE_WAIT);
			dlen++;
		}

		con->rcv_next += dlen;
		con->snd_una += acked;
		tcp_senddata(con, TH_ACK);
		break;
	}

	return;

 kill:
	honeyd_log_probe(IP_PROTO_TCP, &honeyd_tmp.conhdr, pktlen,
	    tcp->th_flags);

	/* Do not kill on reset */
	if (tcp->th_flags & TH_RST)
		return;

	syslog(LOG_INFO, "Killing %s connection: tcp %s",
	    (tcp->th_flags & TH_SYN) ? "attempted" : "unknown",
	    honeyd_contoa(&honeyd_tmp.conhdr));

	/* Fake connection element */
	honeyd_tmp.rcv_next = ntohl(tcp->th_seq) + 1;
	honeyd_tmp.snd_una = ntohl(tcp->th_ack);
	honeyd_tmp.tmpl = tmpl;

	/* Even though options processing does not make any sense on 
	 * RST segment, some stacks apparently do it anyway.
	 */
	tcp_do_options(&honeyd_tmp, tcp);

	tcp_send(&honeyd_tmp, TH_RST|TH_ACK, NULL, 0);
	return;

 close:
	if (tiflags & TH_RST) {
		syslog(LOG_INFO, "Connection dropped by reset: tcp %s",
		    honeyd_contoa(&con->conhdr));
	}
	goto free;

 dropwithreset:
	syslog(LOG_INFO, "Connection dropped with reset: tcp %s",
	    honeyd_contoa(&con->conhdr));
	tcp_send(con, TH_RST|TH_ACK, NULL, 0);
 free:
	tcp_free(con);
	return;
 closed:
	syslog(LOG_INFO, "Connection closed: tcp %s",
	    honeyd_contoa(&con->conhdr));
	/* Forget about this connection */
	tcp_free(con);
 drop:
	return;

 justlog:
	honeyd_settcp(&honeyd_tmp, ip, tcp, 0);
	honeyd_log_probe(IP_PROTO_TCP,&honeyd_tmp.conhdr,pktlen,tcp->th_flags);
}

int
udp_send(struct udp_con *con, u_char *payload, u_int len)
{
	u_char pkt[1500];
	struct udp_hdr *udp;
	u_int iplen;
	uint16_t id = rand_uint16(honeyd_rand);
	int dontfragment = 0;

	/* Statistics */
	con->conhdr.sent += len;

	ip_personality(con->tmpl, &id);

	udp = (struct udp_hdr *)(pkt + IP_HDR_LEN);
	udp_pack_hdr(udp, con->con_dport, con->con_sport, UDP_HDR_LEN + len);

	iplen = IP_HDR_LEN + UDP_HDR_LEN + len;

	/* Src and Dst are reversed both for ip and tcp */
	ip_pack_hdr(pkt, 0, iplen, id,
	    dontfragment ? IP_DF : 0, honeyd_ttl,
	    IP_PROTO_UDP, con->con_ipdst, con->con_ipsrc);

	memcpy(pkt + IP_HDR_LEN + UDP_HDR_LEN, payload, len);

	ip_checksum(pkt, iplen);
	
	if (ip_send(honeyd_ip, pkt, iplen) != iplen)
		syslog(LOG_ERR, "couldn't send packet: %m");

	return (len);
}

void
udp_recv_cb(struct template *tmpl, u_char *pkt, u_short pktlen)
{
	struct ip_hdr *ip = NULL;
	struct udp_hdr *udp;
	struct udp_con *con, honeyd_udp;
	struct addr addr;
	
	uint16_t uh_sum;
	u_char *data;
	u_int dlen;
	u_short portnum;

	ip = (struct ip_hdr *)pkt;

	if (pktlen < (ip->ip_hl << 2) + UDP_HDR_LEN)
		return;

	udp = (struct udp_hdr *)(pkt + (ip->ip_hl << 2));
	data = (u_char *)(pkt + (ip->ip_hl*4) + UDP_HDR_LEN);
	dlen = ntohs(ip->ip_len) - (ip->ip_hl << 2) - UDP_HDR_LEN;
	if ((u_short)dlen != (ntohs(udp->uh_ulen) - UDP_HDR_LEN))
		return;
	
	portnum = ntohs(udp->uh_dport);
	if (honeyd_block(tmpl, IP_PROTO_UDP, portnum))
		goto justlog;

	uh_sum = udp->uh_sum;
	if (uh_sum) {
		ip_checksum(ip, pktlen);
		if (uh_sum != udp->uh_sum)
			goto justlog;
	}

	honeyd_setudp(&honeyd_udp, ip, udp, 0);
	con = (struct udp_con *)SPLAY_FIND(utree, &udpcons, &honeyd_udp.conhdr);

	if (con == NULL) {
		/* Send unreachable on closed port */
		if (!honeyd_portopen(tmpl, IP_PROTO_UDP, portnum)) {
			syslog(LOG_INFO, "Connection to closed port: udp %s",
			    honeyd_contoa(&honeyd_udp.conhdr));
			goto closed;
		}

		/* Otherwise create a new udp connection */
		syslog(LOG_INFO, "Connection: udp %s",
		    honeyd_contoa(&honeyd_udp.conhdr));

		/* Out of memory is dealt by having the port closed */
		if ((con = udp_new(ip, udp, 0)) == NULL) {
			goto closed;
		}

		generic_connect(&con->conhdr, &con->cmd, con);
	}

	/* Keep this state active */
	generic_timeout(&con->timeout, HONEYD_UDP_WAIT);
	con->softerrors = 0;

	/* Statistics */
	con->conhdr.received += dlen;

	/* Add the data to the incoming buffers */
	udp_add_readbuf(con, data, dlen);
	return;

 closed:
	honeyd_log_probe(IP_PROTO_UDP, &honeyd_udp.conhdr, pktlen, 0);

	addr_pack(&addr, ADDR_TYPE_IP, IP_ADDR_BITS, &ip->ip_dst, IP_ADDR_LEN);
	icmp_error_send(&addr, ICMP_UNREACH, ICMP_UNREACH_PORT, ip); 
	return;

 justlog:
	honeyd_setudp(&honeyd_udp, ip, udp, 0);
	honeyd_log_probe(IP_PROTO_UDP, &honeyd_udp.conhdr, pktlen, 0);
}

void
icmp_recv_cb(struct template *tmpl, u_char *pkt, u_short pktlen)
{
	struct ip_hdr *ip = NULL;
	struct icmp_hdr *icmp;
	struct icmp_msg_quote *icmp_quote;
	struct ip_hdr *rip, tmpip;
	struct udp_hdr *udp, tmpudp;
	struct udp_con *con, honeyd_udp;
	/* YM - ICMP Messages */
	struct icmp_msg_echo *icmp_echo;
	struct icmp_msg_timestamp *icmp_tstamp;
	struct icmp_msg_idseq *icmp_idseq;
	struct xp_fingerprint *xp_print = NULL;  /* JVR */
	struct tuple icmphdr;
	struct addr src, dst;
	char asrc[20], adst[20];
	u_char *dat;
	uint16_t cksum;
	int dlen;

	ip = (struct ip_hdr *)pkt;

	if (pktlen < (ip->ip_hl << 2) + ICMP_HDR_LEN)
		return;

	icmp = (struct icmp_hdr *)(pkt + (ip->ip_hl << 2));

	icmphdr.ip_src = ip->ip_src;
	icmphdr.ip_dst = ip->ip_dst;
	icmphdr.type = SOCK_RAW;
	icmphdr.sport = icmp->icmp_type; /* XXX - horrible cludge */
	icmphdr.dport = icmp->icmp_code;
	honeyd_log_probe(IP_PROTO_ICMP, &icmphdr, pktlen, 0);

	/* We can block ICMP, too */
	if (tmpl && tmpl->icmp.status == PORT_BLOCK)
		return;

	if (tmpl != NULL)
		xp_print = tmpl->person->xp_fprint;

	/* Without xprobe fingerprint, we understand only ECHO and UNREACH */
	if (xp_print == NULL) {
		if (!(icmp->icmp_type == ICMP_ECHO) &&
		    !(icmp->icmp_type == ICMP_UNREACH &&
			icmp->icmp_code == ICMP_UNREACH_PORT))
			return;
	}

	cksum = icmp->icmp_cksum;
	ip_checksum(ip, pktlen);
	if (cksum != icmp->icmp_cksum)
		return;

	dlen = pktlen - IP_HDR_LEN - ICMP_HDR_LEN;
	dlen -= 4;

	if (dlen < 0)
		return;

	/* AscII representation of addresses */
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &ip->ip_src, IP_ADDR_LEN);
	addr_pack(&dst, ADDR_TYPE_IP, IP_ADDR_BITS, &ip->ip_dst, IP_ADDR_LEN);
	addr_ntop(&src, asrc, sizeof(asrc));
	addr_ntop(&dst, adst, sizeof(adst));

	switch (icmp->icmp_type) {
	case ICMP_ECHO:
	        icmp_echo = (struct icmp_msg_echo *)(icmp + 1);
		dat = (u_char *)(icmp_echo + 1);
	  
		syslog(LOG_INFO, "Sending ICMP Echo Reply: %s -> %s",
		    adst, asrc);
	        if (xp_print) {
			/* ym: Use our own icmp echo reply function */
			icmp_echo_reply(ip, xp_print->flags.icmp_echo_code,
			    xp_print->flags.icmp_echo_ip_id ? rand_uint16(honeyd_rand): 0,
			    xp_print->flags.icmp_echo_tos_bits ? ip->ip_tos : 0,
			    xp_print->flags.icmp_echo_df_bit ? IP_DF : 0,
			    xp_print->ttl_vals.icmp_echo_reply_ttl.ttl_val,
			    dat, dlen);
		} else {
			icmp_echo_reply(ip, ICMP_CODE_NONE, rand_uint16(honeyd_rand),
			    0, 0, honeyd_ttl, dat, dlen);
		}
		break;

	case ICMP_UNREACH:
		/* Only port unreachable at the moment */
		icmp_quote = (struct icmp_msg_quote *)(icmp + 1);
		rip = (struct ip_hdr *)(&icmp_quote->icmp_ip);

		if (rip->ip_p != IP_PROTO_UDP)
			break;

		udp = (struct udp_hdr *)((u_char *)rip + (ip->ip_hl<<2));
		tmpip.ip_src = rip->ip_dst;
		tmpip.ip_dst = rip->ip_src;
		tmpudp.uh_sport = udp->uh_dport;
		tmpudp.uh_dport = udp->uh_sport;
		honeyd_setudp(&honeyd_udp, &tmpip, &tmpudp, 0);

		/* Find matching state */
		con = (struct udp_con *)SPLAY_FIND(utree, &udpcons,
		    &honeyd_udp.conhdr);
		if (con == NULL)
			break;

		con->softerrors++;
		syslog(LOG_INFO,
		    "Received port unreachable: %s -> %s: errors %d",
		    asrc, adst, con->softerrors);
		if (con->softerrors >= HONEYD_MAX_SOFTERRS)
			udp_free(con);

		break;

		/* YM: Add ICMP Timestamp reply capability */
	case ICMP_TSTAMP:
	        if (xp_print->flags.icmp_timestamp_reply) {
			icmp_tstamp = (struct icmp_msg_timestamp *)
			    ((u_char*)pkt + (ip->ip_hl << 2));
		    
			syslog(LOG_INFO, "Sending ICMP Timestamp Reply: %s -> %s",
			    adst, asrc);
		    
			icmp_timestamp_reply(ip, icmp_tstamp,
			    xp_print->ttl_vals.icmp_timestamp_reply_ttl.ttl_val);
		}
	        break;

		/* YM: Added ICMP Address Mask reply capability */
	case ICMP_MASK:
	        if (xp_print->flags.icmp_addrmask_reply) {
			icmp_idseq = (struct icmp_msg_idseq *)(icmp + 1);
		    
			syslog(LOG_INFO, "Sending ICMP Address Mask Reply: %s -> %s",
			    adst, asrc);
		
			icmp_mask_reply(ip, icmp_idseq,
			    xp_print->ttl_vals.icmp_addrmask_reply_ttl.ttl_val,
			    HONEYD_ADDR_MASK);
		}
		break;

		/* YM: Added ICMP Information reply capability */
	case ICMP_INFO:
	        if (xp_print->flags.icmp_info_reply) {
			icmp_idseq = (struct icmp_msg_idseq *)(icmp + 1);
		    
			syslog(LOG_INFO, "Sending ICMP Info Reply: %s -> %s",
			    adst, asrc);
		
			icmp_info_reply(ip, icmp_idseq,
			    xp_print->ttl_vals.icmp_info_reply_ttl.ttl_val);
		}
		break;

	default:
		break;
	}
}

void
honeyd_dispatch(struct template *tmpl, struct ip_hdr *ip)
{
	struct tuple iphdr;

	switch(ip->ip_p) {
	case IP_PROTO_TCP:
		tcp_recv_cb(tmpl, (u_char *)ip, ntohs(ip->ip_len));
		break;
	case IP_PROTO_UDP:
		udp_recv_cb(tmpl, (u_char *)ip, ntohs(ip->ip_len));
		break;
	case IP_PROTO_ICMP:
		icmp_recv_cb(tmpl, (u_char *)ip, ntohs(ip->ip_len));
		break;
	default:
		iphdr.ip_src = ip->ip_src;
		iphdr.ip_dst = ip->ip_dst;
		iphdr.type = -1;
		honeyd_log_probe(ip->ip_p, &iphdr, ntohs(ip->ip_len), 0);
		return;
	}
}

void
honeyd_recv_direct(struct template *tmpl, struct ip_hdr *ip)
{
	uint16_t ipoff;

	/* Check for fragmentation */
	ipoff = ntohs(ip->ip_off);
	if ((ipoff & IP_OFFMASK) || (ipoff & IP_MF)) {
		ip_fragment(tmpl, ip);
		return;
	}

	honeyd_dispatch(tmpl, ip);
	return;
}

void
honeyd_delay_cb(int fd, short which, void *arg)
{
	struct delay *delay = arg;
	struct ip_hdr *ip = (struct ip_hdr *)(delay + 1);

	honeyd_ttl = delay->ttl;
	if (!ip->ip_ttl)
		icmp_error_send(&delay->host,
		    ICMP_TIMEXCEED, ICMP_TIMEXCEED_INTRANS, ip);
	else
		honeyd_recv_direct(delay->tmpl, ip);

	free(delay);
}

/* Delays a packet for a specified amount of ms to simulate routing delay */

void
honeyd_delay_packet(struct template *tmpl, struct ip_hdr *ip,
    struct addr *host, int ms)
{
	struct delay *delay;
	struct timeval tv;
	u_int iplen = ntohs(ip->ip_len);

	if ((delay = malloc(sizeof(struct delay) + iplen)) == NULL) {
		syslog(LOG_WARNING, "%s: malloc: %m", __FUNCTION__);
		return;
	}
	
	if (host != NULL)
		delay->host = *host;
	delay->tmpl = tmpl;
	delay->ttl = honeyd_ttl;

	timeout_set(&delay->timeout, honeyd_delay_cb, delay);
	timerclear(&tv);
	tv.tv_sec = ms / 1000;
	tv.tv_usec = (ms % 1000) * 1000;
	timeout_add(&delay->timeout, &tv);

	memcpy(delay + 1, ip, iplen);
}


void
honeyd_recv_cb(u_char *u, const struct pcap_pkthdr *pkthdr, const u_char *pkt)
{
	struct ip_hdr *ip;
	struct template *tmpl;
	struct addr addr;
	struct addr host;
	struct router *r;
	struct router_entry *rte;
	char gw[30];
	double packetloss = 1;
	int delay = 0;
	u_short iplen;

	if (pkthdr->caplen < (unsigned int)(honeyd_dloff + IP_HDR_LEN))
		return;

	ip = (struct ip_hdr *)(pkt + honeyd_dloff);

	iplen = ntohs(ip->ip_len);
	if (pkthdr->caplen - honeyd_dloff < iplen)
		return;
	if (ip->ip_hl << 2 > iplen)
		return;
	if (ip->ip_hl << 2 < sizeof(struct ip_hdr))
		return;

	/* Check our own address */
	addr_pack(&addr, ADDR_TYPE_IP, IP_ADDR_BITS, &ip->ip_dst, IP_ADDR_LEN);
	if (addr_cmp(&addr, &honeyd_ifent.intf_addr) == 0)
		return;

	honeyd_ttl = HONEYD_DFL_TTL;

	if (!router_used)
		goto direct;

	host = router_entry;
	
	while (addr_cmp(&host, &addr) != 0 && --ip->ip_ttl) {
		honeyd_ttl--;
		if ((r = router_find(&host)) == NULL) {
		noroute:
			syslog(LOG_DEBUG, "No route to %s", addr_ntoa(&addr));
			return;
		}
		if ((rte = router_find_nexthop(r, &addr)) == NULL)
			goto noroute;
		if (rte->latency)
			delay += rte->latency;
		else
			delay += 3;
		if (rte->packetloss)
			packetloss *= 1 - ((double)rte->packetloss / 5000.0);
		if (rte->type == ROUTE_LINK)
			break;
		host = rte->gw;
	}

	/* Calculate the packet loss rate */
	packetloss = (1 - packetloss) * 10000;
	if (rand_uint16(honeyd_rand) % 10000 < packetloss)
		return;

	if (ip->ip_ttl)
		goto direct;

	addr_ntop(&host, gw, sizeof(gw));
	syslog(LOG_DEBUG, "TTL exceeded for dst %s at gw %s",
	    addr_ntoa(&addr), gw);

	if (delay) {
		honeyd_delay_packet(NULL, ip, &host, 2 * delay);
		return;
	}

	icmp_error_send(&host, ICMP_TIMEXCEED, ICMP_TIMEXCEED_INTRANS, ip);
	return;

 direct:
	tmpl = template_find(addr_ntoa(&addr));
	if (tmpl == NULL)
		tmpl = template_find("default");

	
	if (tmpl != NULL && tmpl->drop_inrate) {
		uint16_t value;
		value = rand_uint16(honeyd_rand) % (100*100);
		if (value < tmpl->drop_inrate)
			return;
	}
	
	if (delay) {
		honeyd_delay_packet(tmpl, ip, NULL, 2 * delay);
		return;
	}
	honeyd_recv_direct(tmpl, ip);
}

void
honeyd_recv(int fd, short type, void *ev)
{
	if (!honeyd_dopoll)
		event_add((struct event *)ev, NULL);

	if (pcap_dispatch(honeyd_pcap, -1, honeyd_recv_cb, NULL) < 0)
		syslog(LOG_ERR, "pcap_dispatch: %s",
		    pcap_geterr(honeyd_pcap));
}
 
void
honeyd_poll_recv(int fd, short type, void *ev)
{
	struct timeval tv = HONEYD_POLL_INTERVAL;

	timeout_add(ev, &tv);

	honeyd_recv(fd, type, ev);
}

void
terminate_handler(int sig)
{
	extern int event_gotsig;

	event_gotsig = 1;
	honeyd_sig = sig;
}

void
child_handler(int sig)
{
#ifndef WIN32
	int s = errno;

	if (signal(SIGCHLD, child_handler) == SIG_ERR) {
		extern int event_gotsig;
		event_gotsig = 1;
		honeyd_sig = SIGTERM;
	}

	while (waitpid(-1, NULL, WNOHANG) > 0)
		honeyd_nchildren--;
	errno = s;
#endif
}

int
honeyd_signal(void)
{
	syslog(LOG_INFO, "exiting on signal %d", honeyd_sig);
	honeyd_exit(0);

	/* NOTREACHED */
	return (0);
}

int
main(int argc, char *argv[])
{
	struct event recv_ev;
	extern int (*event_sigcb)(void);
	char *dev, *config = NULL;
	char *pers = PATH_HONEYDDATA "/nmap.prints";
	char *xprobe = PATH_HONEYDDATA "/xprobe2.conf";
	char *assoc = PATH_HONEYDDATA "/nmap.assoc";
	char *logfile = NULL;
#ifndef WIN32
	int c, debug;
#endif
	FILE *fp;
#ifdef WIN32
	int c;
	char ebuf[PCAP_ERRBUF_SIZE];
	pcap_if_t *alldevs, *d;
	int i;
#endif

	dev = NULL;
	debug = 0;

#ifdef WIN32
	while ((c = getopt(argc, argv, "WVPdi:p:x:a:f:l:h?")) != -1) {
#else
	while ((c = getopt(argc, argv, "VPdi:p:x:a:f:l:h?")) != -1) {
#endif
		switch (c) {
#ifdef WIN32
    case 'W':
				if (pcap_findalldevs(&alldevs, ebuf) == -1)
					errx(1, "pcap_findalldevs: %s", ebuf);

				printf("\nInterface\tDevice\t\tDescription\n-------------------------------------------\n");
				for(i = 1, d=alldevs;d;d=d->next, i++) {
					printf("%d %s",i, d->name);

					if (d->description)
						printf("\t%s",d->description);

					printf("\n");
				}
				exit(1);
      break;
#endif  /* WIN32 */
		case 'V':
			printf("Honeyd Version %s\n", VERSION);
			exit(0);
		case 'P':
			honeyd_dopoll = 1;
			break;
		case 'd':
			debug = 1;
			break;
		case 'i':
			dev = optarg;
			break;
		case 'f':
			config = optarg;
			break;
		case 'l':
			logfile = optarg;
			break;
		case 'x':
			xprobe = optarg;
			break;
		case 'a':
			assoc = optarg;
			break;
		case 'p':
			pers = optarg;
			break;
		default:
			usage();
			break;
		}
	}
	argc -= optind;
	argv += optind;

	if ((honeyd_rand = rand_open()) == NULL)
		err(1, "rand_open");

#ifdef WIN32
	init_winsock();
#endif
	/* Initalize libevent but without kqueue because of bpf */
	setenv("EVENT_NOKQUEUE", "yes", 0);
	event_init();

	config_init();
	personality_init();
	xprobe_personality_init();
	associations_init();

	/* Xprobe2 fingerprints */
	if ((fp = fopen(xprobe, "r")) == NULL)
		err(1, "fopen(%s)", xprobe);
	if (xprobe_personality_parse(fp) == -1)
		errx(1, "parsing xprobe personality file failed");
	fclose(fp);
	
	/* Association between xprobe and nmap fingerprints */
	if ((fp = fopen(assoc, "r")) == NULL)
		err(1, "fopen(%s)", assoc);
	if (parse_associations(fp) == -1)
		errx(1, "parsing associations file failed");
	fclose(fp);

	/* Nmap fingerprints */
	if ((fp = fopen(pers, "r")) == NULL)
		err(1, "fopen(%s)", pers);
	if (personality_parse(fp) == -1)
		errx(1, "parsing personality file failed");
	fclose(fp);

	if (config != NULL) {
		if ((fp = fopen(config, "r")) == NULL)
			err(1, "fopen(%s)", config);
		if (parse_configuration(fp, config) == -1)
			errx(1, "parsing configuration file failed");
		fclose(fp);
	}
	
	if (argc == 0)
		honeyd_init(dev, 0, NULL);
	else
		honeyd_init(dev, argc, argv);
	
	ip_fragment_init();

	if ((fp = fopen(PIDFILE, "w")) == NULL)
		err(1, "fopen");

#ifndef WIN32
	if (!debug) {
		setlogmask(LOG_UPTO(LOG_INFO));
		
		if (daemon(1, 0) < 0) {
			unlink(PIDFILE);
			err(1, "daemon");
		}
	}
	fprintf(fp, "%d\n", getpid());
#else
	fprintf(fp, "%d\n", GetCurrentThreadId());
#endif
	fclose(fp);
	
	chmod(PIDFILE, 0644);

	if (!honeyd_dopoll) {
		event_set(&recv_ev, pcap_fileno(honeyd_pcap), EV_READ,
		    honeyd_recv, &recv_ev);
		event_add(&recv_ev, NULL);
	} else {
		struct timeval tv = HONEYD_POLL_INTERVAL;

		syslog(LOG_INFO, "switching to polling mode");
		timeout_set(&recv_ev, honeyd_poll_recv, &recv_ev);
		timeout_add(&recv_ev,&tv);
	}
#ifndef WIN32
	/* Setup signal handler */
	if (signal(SIGPIPE, SIG_IGN) == SIG_ERR) {
		perror("signal");
		return (-1);
	}
#endif

	if (signal(SIGINT, terminate_handler) == SIG_ERR) {
		perror("signal");
		return (-1);
	}
	if (signal(SIGTERM, terminate_handler) == SIG_ERR) {
		perror("signal");
		return (-1);
	}
#ifndef WIN32
	if (signal(SIGCHLD, child_handler) == SIG_ERR) {
		perror("signal");
		return (-1);
	}
#endif
	event_sigcb = honeyd_signal;
	
	if (logfile != NULL)
		honeyd_logstart(logfile);

	event_dispatch();

	syslog(LOG_ERR, "Kqueue does not recognize bpf filedescriptor.");

#ifdef WIN32
	WSACleanup();
#endif

	return (0);
}
