/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include "config.h"

#ifdef WIN32
#include <winsock2.h>
#else
#include <sys/param.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/queue.h>

#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnet.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "personality.h"

SPLAY_HEAD(templtree, template) templates;

int
templ_compare(struct template *a, struct template *b)
{
	return (strcmp(a->name, b->name));
}

SPLAY_PROTOTYPE(templtree, template, node, templ_compare);

SPLAY_GENERATE(templtree, template, node, templ_compare);

int
port_compare(struct port *a, struct port *b)
{
	int diff;

	diff = a->proto - b->proto;
	if (diff)
		return (diff);

	return ((int)a->number - (int)b->number);
}

SPLAY_PROTOTYPE(porttree, port, node, port_compare);

SPLAY_GENERATE(porttree, port, node, port_compare);

/* Prototypes */
int template_subsystem(struct template *, char *);

void
config_init(void)
{
	SPLAY_INIT(&templates);
}

struct template *
template_find(char *name)
{
	struct template tmp;

	tmp.name = name;
	return (SPLAY_FIND(templtree, &templates, &tmp));
}

int
template_create(char *name)
{
	struct template *tmpl;

	if (template_find(name))
		return (-1);

	if ((tmpl = calloc(1, sizeof(struct template))) == NULL)
		err(1, "%s: calloc", __func__);

	tmpl->name = strdup(name);

	/* UDP ports are closed by default */
	tmpl->udp.status = PORT_RESET;

	SPLAY_INIT(&tmpl->ports);
	SPLAY_INSERT(templtree, &templates, tmpl);

	/* Configured subsystems */
	TAILQ_INIT(&tmpl->subsystems);

	return (0);
}

struct port *
port_find(struct template *tmpl, int proto, int number)
{
	struct port tmpport;
	
	tmpport.proto = proto;
	tmpport.number = number;
	
	return (SPLAY_FIND(porttree, &tmpl->ports, &tmpport));
}

void
port_action_clone(struct action *dst, struct action *src)
{
	*dst = *src;
	if (src->action) {
		dst->action = strdup(src->action);
		if (dst->action == NULL)
			err(1, "%s: strdup", __func__);
	}
}

void
port_free(struct template *tmpl, struct port *port)
{
	SPLAY_REMOVE(porttree, &tmpl->ports, port);

	if (port->sub_conport != NULL) {
		/* Back pointer to connection object.
		 * It allows us to remove the reference to this object
		 * in the connection.
		 * However, at this point we really need to tear down
		 * that connection, too.
		 */
		*port->sub_conport = NULL;
	}

	if (port->sub != NULL)
		TAILQ_REMOVE(&port->sub->ports, port, next);
	if (port->sub_fd != -1)
		close(port->sub_fd);
	if (port->action.action != NULL)
		free (port->action.action);
	if (port->action.aitop != NULL)
#ifdef WIN32
		{
			free(port->action.aitop);
		}
#else
		freeaddrinfo(port->action.aitop);
#endif
	free(port);
}

struct port *
port_insert(struct template *tmpl, int proto, int number,
    struct action *action)
{
	struct port *port, tmpport;
	
	tmpport.proto = proto;
	tmpport.number = number;
	
	if (SPLAY_FIND(porttree, &tmpl->ports, &tmpport) != NULL)
		return (NULL);
	
	if ((port = calloc(1, sizeof(struct port))) == NULL)
		err(1, "%s: calloc", __func__);

	port->sub = NULL;
	port->sub_fd = -1;
	port->proto = proto;
	port->number = number;
	port_action_clone(&port->action, action);
	    
	SPLAY_INSERT(porttree, &tmpl->ports, port);

	return (port);
}

/* Create a random port in a certain range */

struct port *
port_random(struct template *tmpl, int proto, struct action *action,
    int min, int max)
{
	extern rand_t *honeyd_rand;
	struct port *port = NULL;
	int count = 100;
	int number;

	while (count-- && port == NULL) {
		number = rand_uint16(honeyd_rand) % (max - min) + min;
		port = port_insert(tmpl, proto, number, action);
	}

	return (port);
}

int
template_add(struct template *tmpl, int proto, int number,
    struct action *action)
{
	return (port_insert(tmpl, proto, number, action) == NULL ? -1 : 0);
}

int
template_clone(char *newname, struct template *tmpl, int start)
{
	struct subsystem *subsystem;
	struct template *newtmpl;
	struct port *port;
	char *argv[4];

	if (template_create(newname) == -1)
		return (-1);

	if ((newtmpl = template_find(newname)) == NULL)
		return (-1);

	SPLAY_FOREACH(port, porttree, &tmpl->ports) {
		if (port_insert(newtmpl, port->proto, port->number,
			&port->action) == NULL)
			return (-1);
	}

	if (tmpl->person)
		newtmpl->person = personality_clone(tmpl->person);

	port_action_clone(&newtmpl->tcp, &tmpl->tcp);
	port_action_clone(&newtmpl->udp, &tmpl->udp);
	port_action_clone(&newtmpl->icmp, &tmpl->icmp);

	newtmpl->timestamp = tmpl->timestamp;
	newtmpl->uid = tmpl->uid;
	newtmpl->gid = tmpl->gid;
	newtmpl->drop_inrate = tmpl->drop_inrate;

	/* Clone subsystems */
	TAILQ_FOREACH(subsystem, &tmpl->subsystems, next)
		template_subsystem(newtmpl, subsystem->cmdstring);

	if (!start)
		return (0);

	argv[0] = "/bin/sh";
	argv[1] = "-c";
	argv[3] = NULL;

	/* Start background processes */
	TAILQ_FOREACH(subsystem, &newtmpl->subsystems, next) {
		argv[2] = subsystem->cmdstring;
		if (cmd_subsystem(newtmpl, subsystem, "/bin/sh", argv) == -1)
			errx(1, "%s: can not start subsystem \"%s\" for %s",
			    subsystem->cmdstring, newtmpl->name);
	}

	return (0);
}

int
template_subsystem(struct template *tmpl, char *subsystem)
{
	struct subsystem *sub;
	
	if ((sub = calloc(1, sizeof(struct subsystem))) == NULL)
		err(1, "%s: calloc", __func__);

	if ((sub->cmdstring = strdup(subsystem)) == NULL)
		err(1, "%s: strdup", __func__);

	TAILQ_INIT(&sub->ports);
	sub->tmpl = tmpl;
	sub->cmd.pid = -1;

	TAILQ_INSERT_TAIL(&tmpl->subsystems, sub, next);

	return (0);
}
